/*
 Erica Sadun, http://ericasadun.com
 iPhone Developer's Cookbook, 3.0 Edition
 BSD License, Use at your own risk
 */

//
//  TreeNode.m
//  Created by Erica Sadun on 4/6/09.
//

#import "TreeNode.h"

// Makro skracające ciąg tekstowy.
#define STRIP(X)	[X stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceAndNewlineCharacterSet]]

@implementation TreeNode
@synthesize parent, children, key, leafvalue;

#pragma mark Tworzenie i inicjalizacja egzemplarzy TreeNode
+ (TreeNode *) treeNode
{
    TreeNode *node = [[self alloc] init];
    node.children = [NSMutableArray array];
	return node;
}

#pragma mark Typ TreeNode
- (BOOL) isLeaf
{
	return (self.children.count == 0);
}

- (BOOL) hasLeafValue
{
	return (self.leafvalue != nil);
}

#pragma mark Pobieranie danych TreeNode
// Zwrot tablicy kluczy potomnych. Brak rekurencji.
- (NSArray *) keys
{
	NSMutableArray *results = [NSMutableArray array];
	for (TreeNode *node in self.children) [results addObject:node.key];
	return results;
}

// Zwrot tablicy kluczy potomnych z użyciem rekurencji.
- (NSArray *) allKeys
{
	NSMutableArray *results = [NSMutableArray array];
	for (TreeNode *node in self.children)
	{
		[results addObject:node.key];
		[results addObjectsFromArray:node.allKeys];
	}
	return results;
}

- (NSArray *) uniqArray: (NSArray *) anArray
{
	NSMutableArray *array = [NSMutableArray array];
	for (id object in [anArray sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)])
		if (![[array lastObject] isEqualToString:object]) [array addObject:object];
	return array;
}

// Zwrot posortowanej, unikalnej tablicy kluczy potomnych. Brak rekurencji.
- (NSArray *) uniqKeys
{
	return [self uniqArray:[self keys]];
}

// Zwrot posortowanej, unikalnej tablicy kluczy potomnych z użyciem rekurencji.
- (NSArray *) uniqAllKeys
{
	return [self uniqArray:[self allKeys]];
}

// Zwrot tablicy węzłów potomnych. Brak rekurencji.
- (NSArray *) leaves
{
	NSMutableArray *results = [NSMutableArray array];
	for (TreeNode *node in self.children) if (node.leafvalue) [results addObject:node.leafvalue];
	return results;
}

// Zwrot tablicy węzłów potomnych z użyciem rekurencji.
- (NSArray *) allLeaves
{
	NSMutableArray *results = [NSMutableArray array];
	for (TreeNode *node in self.children)
	{
		if (node.leafvalue) [results addObject:node.leafvalue];
		[results addObjectsFromArray:node.allLeaves];
	}
	return results;
}

#pragma mark Wyszukiwanie i pobieranie TreeNode

// Zwrot pierwszego elementu potomnego dopasowanego do klucza. Wyszukiwanie z użyciem rekurencji.
- (TreeNode *) objectForKey: (NSString *) aKey
{
	TreeNode *result = nil;
	for (TreeNode *node in self.children)
		if ([node.key isEqualToString: aKey])
		{
			result = node;
			break;
		}
	if (result) return result;
	for (TreeNode *node in self.children)
	{
		result = [node objectForKey:aKey];
		if (result) break;
	}
	return result;
}

// Zwrot pierwszego elementu węzła, którego klucz został dopasowany. Wyszukiwanie z użyciem rekurencji.
- (NSString *) leafForKey: (NSString *) aKey
{
	TreeNode *node = [self objectForKey:aKey];
	return node.leafvalue;
}

// Zwrot wszystkich elementów potomnych dopasowanych do klucza. Wyszukiwanie z użyciem rekurencji.
- (NSMutableArray *) objectsForKey: (NSString *) aKey
{
	NSMutableArray *result = [NSMutableArray array];
	for (TreeNode *node in self.children)
	{
		if ([node.key isEqualToString: aKey]) [result addObject:node];
		[result addObjectsFromArray:[node objectsForKey:aKey]];
	}
	return result;
}

// Zwrot wszystkich węzłów, których klucze zostały dopasowane. Wyszukiwanie z użyciem rekurencji.
- (NSMutableArray *) leavesForKey: (NSString *) aKey
{
	NSMutableArray *result = [NSMutableArray array];
	for (TreeNode *node in [self objectsForKey:aKey])
		if (node.leafvalue)
			[result addObject:node.leafvalue];
	return result;
}

// Podążanie za ścieżką dopasowanego klucza. Zwracany jest obiekt.
- (TreeNode *) objectForKeys: (NSArray *) keys
{
	if ([keys count] == 0) return self;
	
	NSMutableArray *nextArray = [NSMutableArray arrayWithArray:keys];
	[nextArray removeObjectAtIndex:0];
	
	for (TreeNode *node in self.children)
	{
		if ([node.key isEqualToString:[keys objectAtIndex:0]])
			return [node objectForKeys:nextArray];
	}
	
	return nil;
}

// Podążanie za ścieżką dopasowanego klucza. Zwracany jest węzeł.
- (NSString *) leafForKeys: (NSArray *) keys
{
	TreeNode *node = [self objectForKeys:keys];
	return node.leafvalue;
}

#pragma mark Narzędzia danych wyjściowych
// Wyświetlenie drzewa.
- (void) dumpAtIndent: (int) indent into:(NSMutableString *) outstring
{
	for (int i = 0; i < indent; i++) [outstring appendString:@"--"];
	
	[outstring appendFormat:@"[%2d] Klucz: %@ ", indent, key];
	if (self.leafvalue) [outstring appendFormat:@"(%@)", STRIP(self.leafvalue)];
	[outstring appendString:@"\n"];
	
	for (TreeNode *node in self.children) [node dumpAtIndent:indent + 1 into: outstring];
}

- (NSString *) dump
{
	NSMutableString *outstring = [[NSMutableString alloc] init];
	[self dumpAtIndent:0 into:outstring];
	return outstring;
}

#pragma mark Narzędzia konwersji
// Kiedy jesteś pewny, że masz element nadrzędny wszystkich węzłów, przekształć go na słownik.
- (NSMutableDictionary *) dictionaryForChildren
{
	NSMutableDictionary *results = [NSMutableDictionary dictionary];
	
	for (TreeNode *node in self.children)
		if (node.hasLeafValue) [results setObject:node.leafvalue forKey:node.key];
	
	return results;
}

#pragma mark Invocation Forwarding
// Invocation Forwarding pozwala węzłom na działania jak tablica.
- (id)forwardingTargetForSelector:(SEL)sel
{
	if ([self.children respondsToSelector:sel]) return self.children;
	return nil;
}

// Zgodność z selektorem.
- (BOOL)respondsToSelector:(SEL)aSelector
{
	if ( [super respondsToSelector:aSelector] )	return YES;
	if ([self.children respondsToSelector:aSelector]) return YES;
	return NO;
}

// Możliwość udostępnienia klasy NSArray elementom potomnym.
- (BOOL)isKindOfClass:(Class)aClass
{
	if (aClass == [TreeNode class]) return YES;
	if ([super isKindOfClass:aClass]) return YES;
	if ([self.children isKindOfClass:aClass]) return YES;
	
	return NO;
}

#pragma mark Sprzątanie
- (void) dealloc
{
	self.parent = nil;
	self.children = nil;
	self.key = nil;
	self.leafvalue = nil;
}
@end